#ifndef NCrystal_PointwiseDist_hh
#define NCrystal_PointwiseDist_hh

////////////////////////////////////////////////////////////////////////////////
//                                                                            //
//  This file is part of NCrystal (see https://mctools.github.io/ncrystal/)   //
//                                                                            //
//  Copyright 2015-2020 NCrystal developers                                   //
//                                                                            //
//  Licensed under the Apache License, Version 2.0 (the "License");           //
//  you may not use this file except in compliance with the License.          //
//  You may obtain a copy of the License at                                   //
//                                                                            //
//      http://www.apache.org/licenses/LICENSE-2.0                            //
//                                                                            //
//  Unless required by applicable law or agreed to in writing, software       //
//  distributed under the License is distributed on an "AS IS" BASIS,         //
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  //
//  See the License for the specific language governing permissions and       //
//  limitations under the License.                                            //
//                                                                            //
////////////////////////////////////////////////////////////////////////////////

#include "NCrystal/NCDefs.hh"

namespace NCrystal {
  //Class which provides random sampling of a 1D piece-wise linear
  //distribution function. The function is defined by its non-negative
  //values on a given set of points, which must be specified in
  //ascending order. The integral_weight parameter is only used when
  //merging two distributions.
  class PointwiseDist {
  public:
    PointwiseDist(const VectD &x, const VectD &y, double integral_weight=1.0 );
    ~PointwiseDist();
    //Percentile (argument must be in [0,1]):
    double percentile( double percentile_value ) const { return percentileWithIndex(percentile_value).first; }

    //Sample:
    double sample(RandomBase& rng) const { return percentileWithIndex(rng.generate()).first; }

    PointwiseDist& operator+=(const PointwiseDist&);
    PointwiseDist& operator*=(double frac);
    void setIntegralWeight(double);
    void print() const;

    const VectD& getXVals() const { return m_x; }
    const VectD& getYVals() const { return m_y; }

    //Convenience constructor (would not be needed if we had C++17's std::make_from_tuple):
    PointwiseDist(const std::pair<VectD,VectD>& xy, double integral_weight=1.0 )
      : PointwiseDist(xy.first,xy.second,integral_weight) {}

    //versions which also returns index of bin in which returned value resides
    //(i.e returns (value,idx) where value will lie in interval
    //[getXVals().at(idx),getXVals().at(idx+1)]):
    std::pair<double,unsigned> percentileWithIndex( double percentile_value ) const;
    std::pair<double,unsigned> sampleWithIndex( RandomBase& rng ) const { return percentileWithIndex(rng.generate()); }

  private:
    //todo: We have both m_cdf and m_y, although they essentially contain the
    //same info. Could we implement more light-weight version? Could we
    //implement as a non-owning view, i.e. which keeps m_x in span (but likely
    //needs to be possible to be owning still). Or using shared ptrs?
    VectD m_cdf;
    VectD m_x;
    VectD m_y;
    double m_iweight;
  };
}

#endif
